
// modules

use std::ops;

// types

pub trait Atom: Copy + Sized + Default + ops::Neg<Output = Self> + ops::Add<Output = Self> + ops::AddAssign + ops::Sub<Output = Self> + ops::SubAssign + ops::Mul<Output = Self> + ops::MulAssign {

   const Zero: Self;

   fn from_f32(x: f32) -> Self;
}

pub trait Unit: Copy + Sized + Default + ops::Neg<Output = Self> + ops::Add<Output = Self> + ops::AddAssign + ops::Sub<Output = Self> + ops::SubAssign + ops::Mul<Output = Self> + ops::MulAssign {

   const Zero: Self;

   type Atom: Atom;

   fn read_f32(iter: &mut impl Iterator<Item = f32>) -> Self;
}

#[derive(Clone, Copy, PartialEq, Eq, Default)]
pub struct S2<A: Atom>(pub A, pub A);

// functions

impl Atom for f32 {

   const Zero: Self = 0.0;

   fn from_f32(x: f32) -> Self { x }
}

impl<A: Atom> Unit for A {

   const Zero: Self = A::Zero;

   type Atom = A;

   fn read_f32(iter: &mut impl Iterator<Item = f32>) -> Self {
      let x = iter.next().expect("read error");
      A::from_f32(x)
   }
}

impl<A: Atom> Unit for S2<A> {

   const Zero: Self = Self(A::Zero, A::Zero);

   type Atom = A;

   fn read_f32(iter: &mut impl Iterator<Item = f32>) -> Self {

      let x0 = iter.next().expect("read error");
      let x1 = iter.next().expect("read error");

      Self(A::from_f32(x0),
           A::from_f32(x1))
   }
}

impl<A: Atom> ops::Neg for S2<A> {

   type Output = Self;

   fn neg(self) -> Self::Output {
      Self(-self.0, -self.1)
   }
}

impl<A: Atom> ops::Add for S2<A> {

   type Output = Self;

   fn add(self, rhs: Self) -> Self::Output {

      Self(self.0 + rhs.0,
           self.1 + rhs.1)
   }
}

impl<A: Atom> ops::AddAssign for S2<A> {

   fn add_assign(&mut self, rhs: Self) {
      *self = *self + rhs;
   }
}

impl<A: Atom> ops::Sub for S2<A> {

   type Output = Self;

   fn sub(self, rhs: Self) -> Self::Output {

      Self(self.0 - rhs.0,
           self.1 - rhs.1)
   }
}

impl<A: Atom> ops::SubAssign for S2<A> {

   fn sub_assign(&mut self, rhs: Self) {
      *self = *self - rhs;
   }
}

impl<A: Atom> ops::Mul for S2<A> {

   type Output = Self;

   fn mul(self, rhs: Self) -> Self::Output {

      Self(self.0 * rhs.0,
           self.1 * rhs.1)
   }
}

impl<A: Atom> ops::MulAssign for S2<A> {

   fn mul_assign(&mut self, rhs: Self) {
      *self = *self * rhs;
   }
}

impl<A: Atom> ops::Mul<A> for S2<A> {

   type Output = Self;

   fn mul(self, rhs: A) -> Self::Output {

      Self(self.0 * rhs,
           self.1 * rhs)
   }
}

