
// modules

mod depth;
pub mod eval;
mod order;
pub mod score;
pub mod search;
pub mod see;
mod tt;

use depth::{Depth};
pub use eval::{Eval, Eval_Cache, Eval_Def, Table_Eval};
use order::{Indexer};
pub use score::{Score};
pub use search::{Table_Search};
use tt::{TT};

use crate::prelude::*;
use super::game::{Key, Time_Control};

use std::{fmt, ops};

// types

#[derive(Clone, Copy, PartialEq, Eq)]
pub struct Move_Long(Mv);

pub struct Engine {

   tt: TT,
}

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Default)]
pub struct Ply(pub u8);

pub type Node = u64;
pub type Time = f32;

type Mv = u32;

// functions

pub fn init() {}

impl Move_Long {

   pub fn new(mv: Move, bd: &Board) -> Self {

      let from  = mv.from();
      let to    = mv.to();

      let pc   = mv.piece     (bd);
      let cap  = mv.cap       (bd);
      let prom = mv.auto_queen(bd);

      Self((Mv::from(pc.val())                << 18)
         | (Mv::from(Piece::option_val(cap))  << 15)
         | (Mv::from(Piece::option_val(prom)) << 12)
         | (Mv::from(from.val())              <<  6)
         | (Mv::from(to  .val())              <<  0))
   }

   pub const fn to(self) -> Square {
      Square::from_int(((self.0 >> 0) & 0o77) as u8)
   }

   pub fn piece(self) -> Piece {
      Piece::from_int(((self.0 >> 18) & 0o7) as u8)
   }

   pub fn cap(self) -> Option<Piece> {
      Piece::from_option_int(((self.0 >> 15) & 0o7) as u8)
   }

   pub fn prom(self) -> Option<Piece> {
      Piece::from_option_int(((self.0 >> 12) & 0o7) as u8)
   }

   pub fn is_cap(self) -> bool {
      self.cap().is_some()
   }

   pub fn is_prom(self) -> bool {
      self.prom().is_some()
   }
}

impl From<Move_Long> for Move {

   fn from(mv: Move_Long) -> Self {
      Self::from_int((mv.0 & ((1 << 15) - 1)) as u16)
   }
}

impl Engine {

   pub fn new(global: &'static Global) -> Self {

      Self {
         tt: TT::new(global.tt_bit),
      }
   }

   pub fn resize_tt(&mut self, bit: u8) {
      self.tt.resize(bit);
   }

   pub fn new_game(&mut self) {
      self.tt.clear();
   }

   pub fn search(&mut self, bd: &Board, si: search::Input) -> search::Output {

      debug_assert!(!bd.is_illegal());
      debug_assert!(bd.can_play());

      self.tt.inc_date();

      let mut bd = bd.clone();
      bd.compact();

      let search = &mut search::SG::new(bd, si, &self.tt);
      let so = search.run();

      so
   }
}

impl Ply {

   pub fn val(self) -> u8 {
      self.0
   }

   pub const fn index(self) -> usize {
      self.0 as usize
   }
}

impl ops::Add for Ply {

   type Output = Self;

   fn add(self, rhs: Self) -> Self::Output {
      Self(self.0 + rhs.0)
   }
}

impl ops::AddAssign for Ply {

   fn add_assign(&mut self, rhs: Self) {
      *self = *self + rhs;
   }
}

impl ops::Sub for Ply {

   type Output = Self;

   fn sub(self, rhs: Self) -> Self::Output {
      Self(self.0 - rhs.0)
   }
}

impl ops::SubAssign for Ply {

   fn sub_assign(&mut self, rhs: Self) {
      *self = *self - rhs;
   }
}

impl fmt::Display for Ply {

   fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
      self.0.fmt(f)
   }
}

