
// modules

use crate::prelude::*;

use super::{Move_Long};

use crate::util::prelude::*;

use std::{marker::PhantomData, ops};

// types

#[derive(Clone, Default)]
pub struct Killer {
   data: Array_Grow<Move_Long, 3>,
}

#[derive(Clone)]
pub struct Table<I, T> {
   data: Vec<T>,
   fake: PhantomData<I>,
}

#[derive(Clone, Copy)]
pub struct Stat(Score);

pub trait Indexer: Clone {

   const Size: Index;

   fn index_16(mv: Move, bd: &Board) -> Index;
   fn index_32(mv: Move_Long) -> Index;
}

#[derive(Clone)] pub struct Index_PT;

type Index = u16;
type Score = u16;

// functions

impl Killer {

   pub fn new() -> Self {

      Self {
         data: Default::default(),
      }
   }

   pub fn clear(&mut self) {
      self.data.clear();
   }

   pub fn add(&mut self, mv: Move_Long) {

      let max = 2;

      let pos = if let Some(pos) = find(&self.data, mv) {

         pos

      } else if self.data.size() == max {

         let pos = self.data.size() - 1; // overwrite last move
         self.data[pos] = mv;
         pos

      } else {

         self.data.add(mv);
         self.data.size() - 1
      };

      debug_assert!(self.data[pos] == mv);
      self.data.move_to_front(pos);

      debug_assert!(self.data.size() <= max);
   }

   pub fn iter(&self) -> impl Iterator<Item = Move_Long> {
      self.data.iter().copied()
   }
}

impl<I: Indexer, T: Clone> Table<I, T> {

   pub fn new(x: T) -> Self {

      Self {
         data: vec![x; I::Size as usize],
         fake: Default::default(),
      }
   }
}

impl<I: Indexer, T> ops::Index<Index> for Table<I, T> {

   type Output = T;

   fn index(&self, index: Index) -> &Self::Output {
      &self.data[index as usize]
   }
}

impl<I: Indexer, T> ops::IndexMut<Index> for Table<I, T> {

   fn index_mut(&mut self, index: Index) -> &mut Self::Output {
      &mut self.data[index as usize]
   }
}

impl<I: Indexer, T> ops::Index<(Move, &Board)> for Table<I, T> {

   type Output = T;

   fn index(&self, (mv, bd): (Move, &Board)) -> &Self::Output {
      &self.data[I::index_16(mv, bd) as usize]
   }
}

impl<I: Indexer, T> ops::IndexMut<(Move, &Board)> for Table<I, T> {

   fn index_mut(&mut self, (mv, bd): (Move, &Board)) -> &mut Self::Output {
      &mut self.data[I::index_16(mv, bd) as usize]
   }
}

impl<I: Indexer, T> ops::Index<Move_Long> for Table<I, T> {

   type Output = T;

   fn index(&self, mv: Move_Long) -> &Self::Output {
      &self.data[I::index_32(mv) as usize]
   }
}

impl<I: Indexer, T> ops::IndexMut<Move_Long> for Table<I, T> {

   fn index_mut(&mut self, mv: Move_Long) -> &mut Self::Output {
      &mut self.data[I::index_32(mv) as usize]
   }
}

impl<I: Indexer, T: Clone + Default> Default for Table<I, T> {

   fn default() -> Self {
      Self::new(T::default())
   }
}

impl Stat {

   const Bit: u8 = 14;

   const Min: Score = 0;
   const Mid: Score = 1 << (Self::Bit - 1);
   const Max: Score = 1 <<  Self::Bit;

   pub fn new() -> Self {
      Self(Self::Mid)
   }

   pub fn rand(&mut self, rand: &mut impl Random) {
      self.0 = rand.interval((Self::Mid - 100).into(), (Self::Mid + 100).into()) as u16;
   }

   pub fn good(&mut self, sc: Score, shift: u8) {
      self.0 += mul_shift((Self::Max - self.0).into(), sc.into(), shift);
   }

   pub fn bad(&mut self, sc: Score, shift: u8) {
      self.0 -= mul_shift((self.0 - Self::Min).into(), sc.into(), shift);
   }
}

impl Default for Stat {

   fn default() -> Self {
      Self::new()
   }
}

impl From<Stat> for Score {

   fn from(stat: Stat) -> Self {
      stat.0
   }
}

pub fn sort_caps(list: &mut [Move], bd: &Board) {
   sort_caps_mvv_lva(list, bd);
}

pub fn sort_caps_mvv_lva(list: &mut [Move], bd: &Board) {

   let score = |mv: Move| {

      let pc = mv.piece(bd);
      let cp = mv.cap(bd).unwrap();

      let a = Score::from(pc.val());
      let v = Score::from(cp.val());

      v * 6 + (5 - a) // MVV/LVA
   };

   sort(list, score);
}

pub fn sort_quiets<I: Indexer>(moves: &mut [Move], scores: &mut [Score], bd: &Board, hist_0: &Table<I, Stat>, hist_1: &Table<I, Stat>, hist_2: &Table<I, Stat>) {

   assert!(scores.len() == moves.len());

   let score = |mv| u16::from(hist_0[(mv, bd)])
                  + u16::from(hist_1[(mv, bd)])
                  + u16::from(hist_2[(mv, bd)]);

   fill_scores(scores, moves, score);
   sort_insert(moves, scores);
}

fn sort(moves: &mut [Move], score: impl FnMut(Move) -> Score) {

   let scores = &mut [Score::MIN; 256];

   fill_scores(scores, moves, score);
   sort_insert(moves, scores);
}

fn fill_scores(scores: &mut [Score], moves: &[Move], mut score: impl FnMut(Move) -> Score) {

   let size = moves.len().min(256);
   if size < 2 { return }

   for i in 0 .. size {
      scores[i] = score(moves[i]);
   }
}

pub fn sort_insert<M: Copy, S: Copy + Ord> (moves: &mut [M], scores: &mut [S]) { // decreasing order

   let size = moves.len().min(256);
   if size < 2 { return }

   for i in 1 .. size {

      let move_i  = moves [i];
      let score_i = scores[i];

      let mut j = i;

      while j > 0 && scores[j - 1] < score_i {
         moves [j] = moves [j - 1];
         scores[j] = scores[j - 1];
         j -= 1;
      }

      moves [j] = move_i;
      scores[j] = score_i;
   }
}

pub fn move_to_front(list: &mut impl Array<Item = Move>, good: Move) {

   if let Some(pos) = find(list.as_slice(), good) {
      list.move_to_front(pos);
   }
}

pub fn find<M: Copy + Eq>(list: &[M], good: M) -> Option<u16> {
   list.iter().position(|&m| m == good).map(|pos| pos as u16)
}

impl Indexer for Index_PT {

   const Size: Index = 1 << 9;

   fn index_16(mv: Move, bd: &Board) -> Index {
      (Index::from(mv.piece(bd).val()) << 6)
    | (Index::from(mv.to   ()  .val()) << 0)
   }

   fn index_32(mv: Move_Long) -> Index {
      (Index::from(mv.piece().val()) << 6)
    | (Index::from(mv.to   ().val()) << 0)
   }
}

fn mul_shift(a: u32, b: u32, c: u8) -> u16 {
   debug_assert!(b > 0 && b < (1 << c));
   ((a * b + (1 << (c - 1))) >> c) as u16
}

