
// modules

use super::{Ply};

use std::ops;

// types

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Default)]
pub struct Score(pub i16);

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub enum Kind { Loss, Eval, Win }

pub type Sc = i16;

// functions

impl Score {

   pub const Max:  Self = Self(10_000);
   pub const Min:  Self = Self(-Self::Max.0);
   pub const Inf:  Self = Self(Self::Max.0 + 1);
   pub const None: Self = Self::Inf.neg();

   pub const Eval_Max: Self = Self(Self::Max.0 - 1_000);
   pub const Eval_Min: Self = Self(-Self::Eval_Max.0);

   pub const fn val(self) -> Sc {
      self.0
   }

   pub fn inc(self) -> Self {
      debug_assert!(self < Self::Max);
      Self(self.0 + 1)
   }

   pub fn dec(self) -> Self {
      debug_assert!(self > Self::Min);
      Self(self.0 - 1)
   }

   pub fn is_valid(self) -> bool {
      self.0.abs() <= 10_000
   }

   pub fn is_eval(self) -> bool {
      self.0.abs() <= 9_000
   }

   pub fn is_loss(self) -> bool {
      self.kind() == Kind::Loss
   }

   pub fn is_win(self) -> bool {
      self.kind() == Kind::Win
   }

   pub fn kind(self) -> Kind {

      if self < Self::Eval_Min {
         Kind::Loss
      } else if self > Self::Eval_Max {
         Kind::Win
      } else {
         Kind::Eval
      }
   }

   pub const fn neg(self) -> Self { // HACK for TT
      Self(-self.0)
   }

   pub fn ply(self) -> Ply {
      debug_assert!(!self.is_eval());
      Ply(u8::try_from((Self::Max.0 - self.0.abs()).min(255)).unwrap())
   }

   pub fn add_ply(self, ply: Ply) -> Self {

      debug_assert!(self.is_valid());

      let sc = Sc::from(ply.val());

      if self.0 < Self::Eval_Min.0 - sc {
         Self(self.0 + sc)
      } else if self.0 > Self::Eval_Max.0 + sc {
         Self(self.0 - sc)
      } else {
         self
      }
   }

   pub fn sub_ply(self, ply: Ply) -> Self {

      debug_assert!(self.is_valid());

      let sc = Sc::from(ply.val());

      let res = if self < Self::Eval_Min {
         Self(self.0 - sc)
      } else if self > Self::Eval_Max {
         Self(self.0 + sc)
      } else {
         self
      };

      debug_assert!(res.is_valid());
      res
   }
}

impl From<f32> for Score {

   fn from(x: f32) -> Self {
      debug_assert!(x.abs() <= 80.0);
      Self(from_real(x))
   }
}

impl From<Score> for f32 {

   fn from(sc: Score) -> Self {
      to_real(sc.0)
   }
}

impl ops::Neg for Score {

   type Output = Self;

   fn neg(self) -> Self::Output {
      Self(-self.0)
   }
}

impl ops::Add for Score {

   type Output = Self;

   fn add(self, rhs: Self) -> Self::Output {
      debug_assert!(self.is_eval());
      self + rhs.0
   }
}

impl ops::Add<Sc> for Score {

   type Output = Self;

   fn add(self, rhs: Sc) -> Self::Output {
      debug_assert!(self.is_eval());
      Self(self.0 + rhs)
   }
}

impl ops::AddAssign for Score {

   fn add_assign(&mut self, rhs: Self) {
      *self = *self + rhs;
   }
}

impl ops::AddAssign<Sc> for Score {

   fn add_assign(&mut self, rhs: Sc) {
      *self = *self + rhs;
   }
}

impl ops::Sub for Score {

   type Output = Self;

   fn sub(self, rhs: Self) -> Self::Output {
      debug_assert!(self.is_eval());
      self - rhs.0
   }
}

impl ops::Sub<Sc> for Score {

   type Output = Self;

   fn sub(self, rhs: Sc) -> Self::Output {
      debug_assert!(self.is_eval());
      Self(self.0 - rhs)
   }
}

impl ops::SubAssign for Score {

   fn sub_assign(&mut self, rhs: Self) {
      *self = *self - rhs;
   }
}

impl ops::SubAssign<Sc> for Score {

   fn sub_assign(&mut self, rhs: Sc) {
      *self = *self - rhs;
   }
}

pub fn to_real(sc: i16) -> f32 {
   sc as f32 / 100.0
}

pub fn from_real(x: f32) -> i16 {
   debug_assert!(x.abs() <= 80.0);
   (x * 100.0).round() as i16
}

