
// modules

use super::{Key, Move, Score};

use std::sync::atomic::{self, Ordering::*};

// constants

const Date_Size: u8 = 16;

// types

pub struct Output {

   pub move_: Option<Move>,

   pub min:  Score,
   pub max:  Score,
   pub eval: Option<Score>,

   pub d_min: u8,
   pub d_max: u8,
}

pub struct TT {

   data: Vec<Data>,

   bit:  u8,
   mask: u32,

   date: u8,
}

#[derive(Clone)]
struct Entry {

   lock:  u32,

   move_: Move,
   min:   Score,
   max:   Score,
   eval:  Score,

   pv:    bool,
   date:  u8,
   d_min: u8,
   d_max: u8,
}

#[repr(align(16))]
struct Data {
   d0: atomic::AtomicU64,
   d1: atomic::AtomicU64,
}

// functions

impl TT {

   const Probes: u32 = 3;

   pub fn new(bit: u8) -> Self {

      const { assert!(size_of::<Entry>() == 16) }
      const { assert!(size_of::<Data >() == 16) }

      assert!(bit >= 12);
      let size = 1 << bit;

      let mut data = Vec::with_capacity(size as usize);

      for i in 0 .. size {

         let mut entry = Data::new();
         entry.clear();

         data.push(entry);
      }

      Self {

         data,

         bit,
         mask: size - 1,

         date: 0,
      }
   }

   pub fn resize(&mut self, bit: u8) {
      if bit != self.bit { *self = Self::new(bit) }
   }

   pub fn clear(&mut self) {

      for entry in self.data.iter_mut() {
         entry.clear();
      }
   }

   pub fn inc_date(&mut self) {
      self.date = (self.date + 1) % Date_Size;
   }

   pub fn probe(&self, key: Key) -> Option<Output> {

      let (index, lock) = key.split(self.bit);

      for i in 0 .. Self::Probes {

         let entry = self.entry(index + i).load();

         if entry.lock == lock {

            return Some(Output {

               move_: entry.has_move().then_some(entry.move_),

               min:  entry.min,
               max:  entry.max,
               eval: (entry.eval != Score::None).then_some(entry.eval),

               d_min: entry.d_min,
               d_max: entry.d_max,

            });
         }
      }

      None
   }

   pub fn store(&self, key: Key, move_: Option<Move>, min: Score, max: Score, eval: Option<Score>, depth: u8, pv: bool) {

      debug_assert!(-Score::Inf <= min && min <= max && max <= Score::Inf);
      debug_assert!((min, max) != (-Score::Inf, Score::Inf));

      let date = self.date;

      let (index, lock) = key.split(self.bit);

      let mut bi = None;
      let mut bs = -1;

      for i in 0 .. Self::Probes {

         let ref_ = self.entry(index + i);

         let mut entry = ref_.load();

         if entry.lock == lock { // hash hit

            if let Some(move_) = move_ { entry.move_ = move_ }

            if min != -Score::Inf {
               entry.min   = min;
               entry.d_min = depth;
            }

            if max != Score::Inf {
               entry.max   = max;
               entry.d_max = depth;
            }

            if let Some(eval) = eval {
               entry.eval = eval;
            }

            entry.pv   = pv;
            entry.date = date;

            ref_.store(&entry);

            return;
         }

         let mut sc = 0;

         sc += I(entry.date != date) << 8;
         sc += 235 - i16::from(entry.depth());
         sc += I(!entry.pv) << 4;
         sc += I(!entry.has_move()) << 2;

         debug_assert!(sc >= 0 && sc < (1 << 9));

         if sc > bs {
            bi = Some(i);
            bs = sc;
         }
      }

      let entry = Entry {

         lock,

         move_: move_.unwrap_or(Move::None),
         min,
         max,
         eval: eval.unwrap_or(Score::None),

         pv,
         date,
         d_min: if min != -Score::Inf { depth } else { 0 },
         d_max: if max !=  Score::Inf { depth } else { 0 },
      };

      self.entry(index + bi.unwrap()).store(&entry);
   }

   fn entry(&self, index: u32) -> &Data {
      &self.data[(index & self.mask) as usize]
   }
}

impl Entry {

   const Def: Self = Self {

      lock:  0,

      move_: Move::None,
      min:   Score::Inf.neg(),
      max:   Score::Inf,
      eval:  Score::None,

      pv:    false,
      date:  0,
      d_min: 0,
      d_max: 0,
   };

   fn has_move(&self) -> bool {
      self.move_ != Move::None
   }

   fn depth(&self) -> u8 {
      self.d_min.max(self.d_max)
   }
}

impl Data {

   fn new() -> Self {

      Self {
         d0: 0.into(),
         d1: 0.into(),
      }
   }

   fn clear(&mut self) {
      self.store(&Entry::Def);
   }

   fn load(&self) -> Entry {

      let d1 = self.d1.load(Relaxed);
      let d0 = self.d0.load(Relaxed) ^ d1;

      Entry {

         lock: (d0 >> 0) as u32,

         move_: Move::from_int((d0 >> 32) as u16),
         min:   Score((d1 >> 32) as u16 as i16),
         max:   Score((d1 >> 16) as u16 as i16),
         eval:  Score((d1 >>  0) as u16 as i16),

         pv:    (d1 >> 56) as u8 != 0,
         date:  (d1 >> 48) as u8,
         d_min: (d0 >> 56) as u8,
         d_max: (d0 >> 48) as u8,
      }
   }

   fn store(&self, e: &Entry) {

      let d0 = ((e.d_min       as u64) << 56)
             | ((e.d_max       as u64) << 48)
             | ((e.move_.val() as u64) << 32)
             | ((e.lock        as u64) <<  0)
             ;

      let d1 = ((e.pv                as u64) << 56)
             | ((e.date              as u64) << 48)
             | ((e.min .val() as u16 as u64) << 32)
             | ((e.max .val() as u16 as u64) << 16)
             | ((e.eval.val() as u16 as u64) <<  0)
             ;

      self.d0.store(d0 ^ d1, Relaxed);
      self.d1.store(d1,      Relaxed);
   }
}

fn I(b: bool) -> i16 {
   b.into()
}

