
// modules

use crate::prelude::*;

use super::pawn;

// functions

pub fn is_illegal(bd: &Board) -> bool {

   let sd = bd.turn();
   let xd = sd.opp();

   is_attacked(bd, bd.king(xd), sd)
}

pub fn is_attacked(bd: &Board, to: Square, sd: Side) -> bool {
   is_attacked_aux(bd, to, sd, bd.all())
}

pub fn is_attacked_through_king(bd: &Board, to: Square, sd: Side) -> bool {
   is_attacked_aux(bd, to, sd, bd.all() - bd.piece(Piece::King, sd.opp()))
}

fn is_attacked_aux(bd: &Board, to: Square, sd: Side, blocker: BB) -> bool {

   let table = &bd.global.table_bb;

   for from in pseudo_sliders_to(bd, to, sd) {
      if table.line_is_empty(from, to, blocker) { return true }
   }

   let tmp = bd.piece(Piece::Knight, sd) & table.moves(Piece::Knight, to);
   if !tmp.is_empty() { return true }

   let tmp = bd.piece(Piece::King, sd) & table.moves(Piece::King, to);
   if !tmp.is_empty() { return true }

   let tmp = bd.pawn(sd) & table.pawn_caps_to(sd, to);
   if !tmp.is_empty() { return true }

   false
}

pub fn is_attacked_tos(bd: &Board, tos: BB, sd: Side, blocker: BB) -> bool {

   let table = &bd.global.table_bb;

   // pawns

   let tmp = tos & pawn::caps_froms(bd, sd);
   if !tmp.is_empty() { return true }

   // pieces

   for from in bd.side(sd) - bd.pawn(sd) {

      let pc = bd.square(from).unwrap().0;

      for to in tos & table.moves(pc, from) {
         if table.line_is_empty(from, to, blocker) { return true }
      }
   }

   false
}

pub fn pseudo_sliders_to(bd: &Board, to: Square, sd: Side) -> BB {

   let table = &bd.global.table_bb;

   let mut res = BB::empty();

   res |= bd.bishop_queen(sd) & table.moves(Piece::Bishop, to);
   res |= bd.rook_queen  (sd) & table.moves(Piece::Rook,   to);

   res
}

pub fn move_is_check(mv: Move, bd: &Board) -> bool {
   move_is_check_aux(mv, bd, true, true)
}

fn move_is_check_aux(mv: Move, bd: &Board, disco: bool, prom: bool) -> bool {

   let from = mv.from();
   let to   = mv.to();
   let sd   = bd.turn();
   let king = bd.king(sd.opp());

   let table = &bd.global.table_bb;

   if disco
   && is_pinned_by(king, from, sd, bd)
   && !table.pin_tos(king, from).has(to)
   {
      return true;
   }

   let pc = if prom && let Some(pm) = mv.auto_queen(bd) {
      pm
   } else {
      mv.piece(bd)
   };

   table.attacks_to(pc, sd, king).has(to)
&& table.line_is_empty(king, to, bd.all() - from)

}

pub fn piece_tos(pc: Piece, sd: Side, from: Square, bd: &Board) -> BB {

   debug_assert!(sd == bd.turn()); // for en-passant

   let table = &bd.global.table_bb;

   if pc == Piece::Pawn {

      let mut side_opp = bd.side(sd.opp());
      if let Some(ep) = bd.ep_square() { side_opp.set(ep) }

      (bd.empty() & table.pawn_moves_from(sd, from))
    | (side_opp   & table.pawn_caps_from (sd, from))

   } else {

      table.moves(pc, from)
   }
}

pub fn is_pinned_by(king: Square, from: Square, sd: Side, bd: &Board) -> bool {
   is_pinned_by_aux(king, from, sd, bd, bd.all())
}

pub fn is_pinned_by_aux(king: Square, from: Square, sd: Side, bd: &Board, blocker: BB) -> bool {

   let table = &bd.global.table_bb;

   for pin in pseudo_sliders_to(bd, king, sd)
            & table.beyond(king, from) {

      if table.line_is_empty(king, pin, blocker - from) {
         return true;
      }
   }

   false
}

