
// modules

use crate::prelude::*;

use super::{attack, legal, square, Key, Wing};

use square::{Inc};

use crate::util::{self, prelude::*};

// types

#[derive(Clone)]
pub struct Board {

   pub global: &'static Global,

   piece: [BB; Piece::Size as usize],
   side:  [BB; Side ::Size as usize],
   turn:  Side,

   square: [Option<(Piece, Side)>; Square::Size as usize],

   key:  Key,
   copy: Copy,

   stack: Array_Grow<Key, 200>,
}

#[derive(Clone, Copy, PartialEq, Eq)]
struct Copy {

   ply:  Ply, // recent # moves without conversion
   null: Ply, // recent # moves without null move

   castle:    BB,
   ep_square: Option<Square>,
   last_cap:  Option<Square>,
}

#[derive(Clone, Copy)]
pub struct Undo {

   cap:     Option<(Piece, Square)>,
   is_prom: bool,
   castle:  Option<Wing>,

   copy: Copy,
}

#[derive(Clone, Copy, PartialEq, Eq)]
pub enum Rep {
   Full, // actual repetition
   Null, // repetition straddles a null move
}

type Ply = u16;

// functions

impl Board {

   pub fn start(global: &'static Global) -> Self {
      Self::from_fen("rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w AHah -", global).unwrap()
   }

   pub fn from_fen(s: &str, global: &'static Global) -> Result<Self, ()> {

      use Side::*;

      let mut bd = Self::new(global);

      if let &[sb, st, sc, se, ..] = s.split_whitespace()
                                      .collect::<Vec<_>>()
                                      .as_slice()
      {
         // board

         let mut fl = 0;
         let mut rk = Square::Rank_Size - 1;

         for c in sb.chars() {

            if c == '/' {

               fl = 0;
               rk -= 1;

            } else if c >= '0' && c <= '9' {

               fl += util::char_sub(c, '0') as Inc; // TODO ###

            } else { // piece

               let sd = if c.is_lowercase() { Black } else { White };
               let sq = Square::new(fl, rk)?;
               let pc = Piece::parse(c.to_ascii_uppercase())?;

               if pc == Piece::Pawn && !BB::pawns().has(sq) { return Err(()) }

               bd.add_piece(pc, sd, sq);
               fl += 1;
            }
         }

         if !(fl == Square::File_Size && rk == 0) { return Err(()) }

         // turn

         let turn = match st {
            "w" => White,
            "b" => Black,
             _  => return Err(()),
         };

         if turn != bd.turn() { bd.pass() }

         // castle

         if sc != "-" {

            for c in sc.chars() {

               let sd   = if c.is_lowercase() { Black } else { White };
               let rk   = square::rank_side(0, sd);
               let king = bd.king(sd);

               if king.rank() != rk { return Err(()) }

               let wing = match c.to_ascii_lowercase() {
                  'k' => BB::right(king.file()),
                  'q' => BB::left (king.file()),
                   c  => BB::file(square::file_from_char(c)?) // file name
               };

               let rooks = bd.piece(Piece::Rook, sd) & wing & BB::rank(rk);
               if rooks.is_empty () { return Err(()) }
               if rooks.is_single() { bd.copy.castle |= rooks }
            }
         }

         // en passant

         if se != "-" {
            let ep = se.parse()?;
            bd.set_ep(ep, turn);
         }

         Ok(bd)

      } else {

         Err(())
      }
   }

   fn new(global: &'static Global) -> Self {

      Self {

         global,

         piece: [BB::empty(); Piece::Size as usize],
         side:  [BB::empty(); Side ::Size as usize],
         turn:  Side::White,

         square: [None; Square::Size as usize],

         key: Key::None,

         copy: Copy {

            ply:  0,
            null: 0,

            castle:    BB::empty(),
            ep_square: None,
            last_cap:  None,
         },

         stack: Array_Grow::new(),
      }
   }

   pub fn do_move(&mut self, mv: Move) -> Undo {

      let from = mv.from();
      let to   = mv.to();

      let sd = self.turn();

      let cp    = mv.cap  (self);
      let is_ep = mv.is_ep(self);

      debug_assert!(self.line_is_empty(from, to));

      let mut cap = None;
      let mut is_prom = false;
      let castle = mv.is_castle(self).then(|| Wing::new(from, to));

      let copy = self.push();

      self.copy.castle -= BB::square(from) | BB::square(to);

      let (mut pc, s) = self.square(from).unwrap();
      debug_assert!(s == sd);

      if let Some(cp) = cp { // capture

         assert!(cp != Piece::King); // #

         self.copy.ply  = 0; // conversion
         self.copy.null = 0;
         self.copy.last_cap = Some(to);

         let to = if is_ep { to.ep_opp() } else { to };
         cap = Some((cp, to));
         self.remove_piece(cp, sd.opp(), to);
      }

      if let Some(wing) = castle {

         let (kf, rf) = (from, to);
         let (kt, rt) = self.global.table_rule.king_rook_to(wing, sd);

         self.remove_piece(Piece::Rook, sd, rf);
         self.move_piece  (Piece::King, sd, kf, kt);
         self.add_piece   (Piece::Rook, sd, rt);

      } else {

         self.move_piece(pc, sd, from, to);
      }

      if pc == Piece::Pawn {

         self.copy.ply  = 0; // conversion
         self.copy.null = 0;

         if to.is_prom_side(sd) {

            is_prom = true;

            self.remove_piece(pc, sd, to);
            pc = mv.prom_or_queen();
            self.add_piece   (pc, sd, to);

         } else if to.val() ^ from.val() == 0o02 { // double push

            self.set_ep(to.ep_opp(), sd.opp());
         }

      } else if pc == Piece::King {

         self.copy.castle -= BB::rank_side(0, sd);
      }

      Undo { cap, is_prom, castle, copy }
   }

   pub fn undo_move(&mut self, mv: Move, undo: &Undo) {

      let from = mv.from();
      let to   = mv.to();

      let xd = self.turn();
      let sd = xd.opp();

      if let Some(wing) = undo.castle {

         let (kf, rf) = (from, to);
         let (kt, rt) = self.global.table_rule.king_rook_to(wing, sd);

         self.remove_piece(Piece::Rook, sd, rt);
         self.move_piece  (Piece::King, sd, kt, kf);
         self.add_piece   (Piece::Rook, sd, rf);

      } else {

         let (mut pc, s) = self.square(to).unwrap();
         debug_assert!(s == sd);

         if undo.is_prom {
            self.remove_piece(pc, sd, to);
            pc = Piece::Pawn;
            self.add_piece   (pc, sd, to);
         }

         self.move_piece(pc, sd, to, from);
      }

      if let Some((cp, to)) = undo.cap { // capture
         self.add_piece(cp, xd, to);
      }

      self.pop(undo.copy);
   }

   pub fn do_null(&mut self) -> Undo {

      let copy = self.push();

      self.copy.null = 0;

      Undo {

         cap:     None,
         is_prom: false,
         castle:  None,

         copy,
      }
   }

   pub fn undo_null(&mut self, undo: &Undo) {
      self.pop(undo.copy);
   }

   pub fn compact(&mut self) { // compact repetition stack

      let ply   = self.copy.ply;
      let delta = self.stack.size() - ply;

      if delta != 0 {

         for i in 0 .. ply {
            self.stack[i] = self.stack[i + delta];
         }

         self.stack.truncate(ply);
      }
   }

   fn push(&mut self) -> Copy {

      let copy = self.copy;

      self.stack.push(self.key());

      self.copy.ply  += 1;
      self.copy.null += 1;

      self.copy.ep_square = None;
      self.copy.last_cap  = None;

      self.pass();

      copy
   }

   fn pop(&mut self, copy: Copy) {

      self.pass();
      self.stack.pop();
      self.copy = copy;
   }

   fn move_piece(&mut self, pc: Piece, sd: Side, from: Square, to: Square) {

      debug_assert!(self.line_is_empty(from, to));

      self.remove_piece(pc, sd, from);
      self.add_piece   (pc, sd, to);
   }

   fn remove_piece(&mut self, pc: Piece, sd: Side, sq: Square) {

      debug_assert!(self.piece[pc.index()].has(sq));
      debug_assert!(self.side [sd.index()].has(sq));

      self.piece[pc.index()].clear(sq);
      self.side [sd.index()].clear(sq);

      debug_assert!(self.square[sq.index()] == Some((pc, sd)));
      self.square[sq.index()] = None;

      self.key ^= self.global.table_hash.piece(pc, sd, sq);
   }

   fn add_piece(&mut self, pc: Piece, sd: Side, sq: Square) {

      debug_assert!(!self.piece[pc.index()].has(sq));
      debug_assert!(!self.side [sd.index()].has(sq));

      self.piece[pc.index()].set(sq);
      self.side [sd.index()].set(sq);

      debug_assert!(self.square[sq.index()] == None);
      self.square[sq.index()] = Some((pc, sd));

      self.key ^= self.global.table_hash.piece(pc, sd, sq);
   }

   fn set_ep(&mut self, ep: Square, sd: Side) {

      debug_assert!(self.copy.ep_square.is_none());

      if !BB::is_disjoint(self.pawn(sd), self.global.table_bb.pawn_caps_to(sd, ep)) {
         self.copy.ep_square = Some(ep);
      }
   }

   pub fn pass(&mut self) { // HACK for SNMP
      self.turn = self.turn.opp();
   }

   pub fn is_illegal(&self) -> bool { // can take opponent king
      attack::is_illegal(self)
   }

   pub fn can_play(&self) -> bool {
      legal::has_legal(self)
   }

   pub fn rep(&self) -> Option<Rep> {

      let key  = self.key();
      let size = self.stack.size();

      let mut ply = 4;

      while ply <= self.ply() {

         if self.stack[size - ply] == key {
            return if ply > self.copy.null { Some(Rep::Null) } else { Some(Rep::Full) };
         }

         ply += 2;
      }

      None
   }

   pub const fn turn(&self) -> Side {
      self.turn
   }

   pub fn piece(&self, pc: Piece, sd: Side) -> BB {
      self.piece[pc as usize] & self.side[sd as usize]
   }

   pub fn no_pawn(&self, sd: Side) -> BB {
      self.side(sd) - self.pawn(sd)
   }

   pub fn pawn(&self, sd: Side) -> BB {
      self.piece(Piece::Pawn, sd)
   }

   pub fn pieces(&self, sd: Side) -> BB {
      self.side(sd) - self.piece(Piece::King, sd) - self.pawn(sd)
   }

   pub fn bishop_queen(&self, sd: Side) -> BB {
      self.piece(Piece::Bishop, sd)
    | self.piece(Piece::Queen,  sd)
   }

   pub fn rook_queen(&self, sd: Side) -> BB {
      self.piece(Piece::Rook,  sd)
    | self.piece(Piece::Queen, sd)
   }

   pub fn is_lone_king(&self, sd: Side) -> bool {
      self.pieces(sd).is_empty()
   }

   pub fn king(&self, sd: Side) -> Square {

      let kings = self.piece(Piece::King, sd);
      debug_assert!(kings.is_single());
      kings.first()
   }

   pub const fn side(&self, sd: Side) -> BB {
      self.side[sd as usize]
   }

   pub fn all(&self) -> BB {
      self.side(Side::White) | self.side(Side::Black)
   }

   pub fn empty(&self) -> BB {
      BB::full() - self.all()
   }

   pub const fn square(&self, sq: Square) -> Option<(Piece, Side)> {
      self.square[sq.index()]
   }

   pub fn square_is_empty(&self, sq: Square) -> bool {
      !self.all().has(sq)
   }

   pub const fn square_is_piece(&self, sq: Square, pc: Piece) -> bool {
      self.piece[pc.index()].has(sq)
   }

   pub const fn square_is_side(&self, sq: Square, sd: Side) -> bool {
      self.side(sd).has(sq)
   }

   pub fn line_is_empty(&self, from: Square, to: Square) -> bool {
      self.global.table_bb.line_is_empty(from, to, self.all())
   }

   pub fn key(&self) -> Key {

      let table = &self.global.table_hash;

      let mut res = self.key;

      res ^= table.turn(self.turn());

      for rook in self.castle() {
         res ^= table.castle(rook);
      }

      if let Some(ep) = self.ep_square() {
         res ^= table.ep(ep);
      }

      res
   }

   pub const fn ply(&self) -> Ply {
      self.copy.ply
   }

   pub fn castle(&self) -> BB {
      self.copy.castle
   }

   pub fn castle_side(&self, sd: Side) -> BB {
      self.castle() & self.side(sd)
   }

   pub const fn ep_square(&self) -> Option<Square> {
      self.copy.ep_square
   }

   pub const fn last_cap(&self) -> Option<Square> {
      self.copy.last_cap
   }
}

