
// modules

use crate::prelude::*;

use crate::util::prelude::*;
use crate::util::random;

use std::ops;

// types

pub struct Table_Hash {
   piece:  [[[Bits; Square::Size as usize]; Piece::Size as usize]; Side::Size as usize],
   turn:   [Bits; Side  ::Size as usize],
   castle: [Bits; Square::Size as usize],
   ep:     [Bits; Square::Size as usize],
}

#[derive(Clone, Copy, PartialEq, Eq, Default)]
pub struct Key(Bits);

type Bits = u64;

// functions

impl Key {

   pub const None: Self = Self(0);

   pub fn split(self, size: u8) -> (u32, u32) { // (index, lock)
      ((self.0 >> (64 - size)) as u32,
       (self.0 >> (32 - size)) as u32)
   }
}

impl ops::BitXor for Key {

   type Output = Self;

   fn bitxor(self, rhs: Self) -> Self {
      Self(self.0 ^ rhs.0)
   }
}

impl ops::BitXorAssign for Key {

   fn bitxor_assign(&mut self, rhs: Self) {
      *self = *self ^ rhs;
   }
}

impl Table_Hash {

   pub fn new() -> Self {

      let rand = &mut random::Random_Phi::new();

      let mut piece  = [[[0; Square::Size as usize]; Piece::Size as usize]; Side::Size as usize];
      let mut turn   = [0; Side  ::Size as usize];
      let mut castle = [0; Square::Size as usize];
      let mut ep     = [0; Square::Size as usize];

      for sd in Side::iter() {
         for pc in Piece::iter() {
            for sq in Square::iter() {
               piece[sd.index()][pc.index()][sq.index()] = rand.next_u64();
            }
         }
      }

      for sd in Side::iter() {
         turn[sd.index()] = rand.next_u64();
      }

      for sq in Square::iter() {
         castle[sq.index()] = rand.next_u64();
         ep    [sq.index()] = rand.next_u64();
      }

      Table_Hash { piece, turn, castle, ep }
   }

   pub const fn piece(&self, pc: Piece, sd: Side, sq: Square) -> Key {
      Key(self.piece[sd.index()][pc.index()][sq.index()])
   }

   pub const fn turn(&self, sd: Side) -> Key {
      Key(self.turn[sd.index()])
   }

   pub const fn castle(&self, sq: Square) -> Key {
      Key(self.castle[sq.index()])
   }

   pub const fn ep(&self, sq: Square) -> Key {
      Key(self.ep[sq.index()])
   }
}

