
// modules

pub mod attack;
pub mod bb;
pub mod board;
pub mod game;
pub mod gen_;
pub mod hash;
pub mod legal;
pub mod pawn;
pub mod rule;
pub mod square;

pub use bb::{BB, Table_BB};
pub use board::{Board, Undo};
pub use hash::{Key, Table_Hash};
pub use legal::{Legal_Info};
pub use rule::{Rule, Table_Rule, Wing};
pub use square::{Colour, Square};

use crate::prelude::*;

use crate::engine::see;

use std::{fmt, mem, ops, str::FromStr, sync::atomic::Ordering};

// types

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub enum Piece { Pawn, Knight, Bishop, Rook, Queen, King }

#[derive(Clone, Copy, PartialEq, Eq)]
pub enum Side { White, Black }

#[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub struct Move(Mv);

#[derive(Clone, Copy)]
pub struct Time_Control {
   pub moves: Option<u16>,
   pub time:  Time,
   pub inc:   Time,
}

type Mv = u16;

// functions

pub fn init() {

   const { assert!(size_of::<BB    >() == 8) }
   const { assert!(size_of::<Move  >() == 2) }
   const { assert!(size_of::<Piece >() == 1) }
   const { assert!(size_of::<Side  >() == 1) }
   const { assert!(size_of::<Square>() == 1) }
}

impl Piece {

   pub const Size: u8 = 6;

   pub const fn from_int(i: u8) -> Self {
      debug_assert!(i < Self::Size);
      unsafe { mem::transmute(i) }
   }

   pub fn from_option_int(i: u8) -> Option<Self> {
      debug_assert!(i <= Self::Size);
      (i != 0).then(|| Self::from_int(i - 1))
   }

   pub const fn val(self) -> u8 {
      self as u8
   }

   pub fn option_val(pc: Option<Self>) -> u8 {
      pc.map_or(0, |pc| pc.val() + 1)
   }

   pub const fn index(self) -> usize {
      self.val() as usize
   }

   pub fn is_king(self) -> bool {
      self == Self::King
   }

   pub fn is_slider(self) -> bool {
      self >= Self::Bishop && self <= Self::Queen
   }

   pub fn parse(c: char) -> Result<Self, ()> {

      match c {
         'P' | 'p' => Ok(Self::Pawn),
         'N' | 'n' => Ok(Self::Knight),
         'B' | 'b' => Ok(Self::Bishop),
         'R' | 'r' => Ok(Self::Rook),
         'Q' | 'q' => Ok(Self::Queen),
         'K' | 'k' => Ok(Self::King),
         _ => Err(()),
      }
   }

   pub fn iter() -> impl Iterator<Item = Self> {
      (0 .. Self::Size).map(|i| unsafe { mem::transmute(i) })
   }

   pub fn no_pawn() -> impl Iterator<Item = Self> {
      (1 .. Self::Size).map(|i| unsafe { mem::transmute(i) })
   }
}

impl fmt::Display for Piece {

   fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {

      use Piece::*;

      let s = match self {
         Pawn   => "P",
         Knight => "N",
         Bishop => "B",
         Rook   => "R",
         Queen  => "Q",
         King   => "K",
      };

      s.fmt(f)
   }
}

impl Side {

   pub const Size: u8 = 2;

   pub const fn val(self) -> u8 {
      self as u8
   }

   pub const fn index(self) -> usize {
      self.val() as usize
   }

   pub const fn opp(self) -> Self {

      match self {
         Self::White => Self::Black,
         Self::Black => Self::White,
      }
   }

   pub const fn is_canon(self) -> bool {
      matches!(self, Self::White)
   }

   pub fn sign<T: ops::Neg<Output = T>>(self, x: T) -> T {

      match self {
         Self::White =>  x,
         Self::Black => -x,
      }
   }

   pub fn iter() -> impl Iterator<Item = Self> {
      (0 .. Self::Size).map(|i| unsafe { mem::transmute(i) })
   }
}

impl Move {

   pub const None: Self = Self(0); // HACK for TT

   pub fn new(from: Square, to: Square) -> Self {

      Self((Mv::from(from.val()) << 6)
         | (Mv::from(to  .val()) << 0))
   }

   pub fn with_prom(from: Square, to: Square, prom: Piece) -> Self {

      debug_assert!(prom >= Piece::Knight && prom <= Piece::Queen);

      Self((Mv::from(Piece::option_val(Some(prom))) << 12)
         | (Mv::from(from.val())                    <<  6)
         | (Mv::from(to  .val())                    <<  0))
   }

   pub fn normalise(self, bd: &Board) -> Self {

      if let Some(prom) = self.auto_queen(bd) {
         Self::with_prom(self.from(), self.to(), prom)
      } else {
         self
      }
   }

   pub const fn from_int(i: Mv) -> Self {
      Self(i & ((1 << 15) - 1))
   }

   pub const fn val(self) -> Mv {
      self.0
   }

   pub const fn from(self) -> Square {
      Square::from_int(((self.0 >> 6) & 0o77) as u8)
   }

   pub const fn to(self) -> Square {
      Square::from_int(((self.0 >> 0) & 0o77) as u8)
   }

   fn prom(self) -> Option<Piece> {
      Piece::from_option_int(((self.0 >> 12) & 0o7) as u8)
   }

   pub fn is_cap(self, bd: &Board) -> bool {
      self.is_ep(bd) ||
      (!bd.square_is_empty(self.to()) && !self.is_castle(bd))
   }

   pub fn is_ep(self, bd: &Board) -> bool {
      self.is_pawn(bd) && bd.ep_square() == Some(self.to())
   }

   pub fn is_prom(self, bd: &Board) -> bool {
      self.is_pawn(bd) && self.to().is_prom()
   }

   pub fn is_push(self, bd: &Board) -> bool {

      self.is_pawn(bd) &&
      self.to().is_seventh(bd.turn()) &&
      !self.is_cap(bd)
   }

   pub fn is_check(self, bd: &Board) -> bool {
      attack::move_is_check(self, bd)
   }

   pub fn is_pawn(self, bd: &Board) -> bool {
      bd.square_is_piece(self.from(), Piece::Pawn)
   }

   pub fn is_king(self, bd: &Board) -> bool {
      bd.square_is_piece(self.from(), Piece::King)
   }

   pub fn is_king_cap(self, bd: &Board) -> bool {
      self.cap(bd) == Some(Piece::King)
   }

   pub fn is_castle(self, bd: &Board) -> bool {
      self.is_king(bd) && bd.square_is_side(self.to(), bd.turn())
   }

   pub fn is_legal(self, bd: &Board) -> bool {

      debug_assert!(!bd.is_illegal());

      let li = Legal_Info::new(bd);
      li.is_pseudo(self, bd) && li.is_legal(self, bd)
   }

   pub fn is_recap(self, bd: &Board) -> bool {
      bd.last_cap() == Some(self.to())
   }

   pub fn is_unique(self, bd: &Board) -> bool {
      debug_assert!(self.is_cap(bd));
      gen_::count_caps_to(bd, BB::full(), self.to()) <= 1
   }

   pub fn is_safe(self, bd: &Board) -> bool {
      see::move_is_safe(self, bd)
   }

   pub fn is_win(self, bd: &Board) -> bool {
      see::move_is_win(self, bd)
   }

   pub fn king_to(self, bd: &Board) -> Square {

      if self.is_castle(bd) {
         let wing = Wing::new(self.from(), self.to());
         bd.global.table_rule.king_rook_to(wing, bd.turn()).0
      } else {
         self.to()
      }
   }

   pub const fn piece(self, bd: &Board) -> Piece {
      bd.square(self.from()).unwrap().0
   }

   pub fn cap(self, bd: &Board) -> Option<Piece> {

      if self.is_castle(bd) {
         None
      } else if self.is_ep(bd) {
         Some(Piece::Pawn)
      } else {
         bd.square(self.to()).map(|(cp, xd)| cp)
      }
   }

   pub fn auto_queen(self, bd: &Board) -> Option<Piece> {
      self.is_prom(bd).then(|| self.prom_or_queen())
   }

   pub fn prom_or_queen(self) -> Piece {
      self.prom().unwrap_or(Piece::Queen)
   }

   pub fn to_uci(self, bd: &Board) -> String {

      let mv = if bd.global.rule.convert.load(Ordering::Relaxed) {
         rule::conv_out(self, bd)
      } else {
         self
      };

      let mut res = String::new();

      res += &format!("{}{}", mv.from(), mv.to());

      if let Some(prom) = mv.auto_queen(bd) {
         res += &prom.to_string().to_lowercase();
      }

      res
   }

   pub fn parse(s: &str, bd: &Board) -> Result<Self, ()> {

      let mv = Self::from_str(s)?;

      Ok(if bd.global.rule.convert.load(Ordering::Relaxed) {
         rule::conv_in(mv, bd)
      } else {
         mv
      })
   }
}

impl FromStr for Move {

   type Err = ();

   fn from_str(s: &str) -> Result<Self, Self::Err> {

      if s.is_ascii() && s.len() >= 4 { // HACK

         let from = s[0 .. 2].parse()?;
         let to   = s[2 .. 4].parse()?;

         let mut iter = s[4 ..].chars();

         if let Some(c) = iter.next() {
            let prom = Piece::parse(c.to_ascii_uppercase())?;
            Ok(Self::with_prom(from, to, prom))
         } else {
            Ok(Self::new(from, to))
         }

      } else {

         Err(())
      }
   }
}

pub fn line_to_uci(line: &[Move], bd: &Board) -> String {

   let mut res = String::new();

   let mut bd = bd.clone();
   bd.compact();

   for &mv in line {

      if !res.is_empty() { res += " " }
      res += &mv.to_uci(&bd);

      bd.do_move(mv);
      bd.compact();
   }

   res
}

