
// modules

use super::square::{self, Inc};

use crate::prelude::*;

use std::{mem, sync::atomic};

// types

pub struct Rule {
   pub convert: atomic::AtomicBool, // KxR <-> K move for castling
}

pub struct Table_Rule {
   castle: [[Castle<Square>; Wing::Size as usize]; Side::Size as usize],
}

#[derive(Clone, Copy)]
struct Castle<T> {
   pub king_to: T,
   pub rook_to: T,
}

#[derive(Clone, Copy, PartialEq, Eq)]
pub enum Wing { A, H }

// variables

static Castle: [[Castle<Inc>; Wing::Size as usize]; 2] = [

   [Castle { king_to: 1, rook_to: 3 },
    Castle { king_to: 6, rook_to: 4 }],

   [Castle { king_to: 2, rook_to: 3 },
    Castle { king_to: 6, rook_to: 5 }],
];

// functions

impl Wing {

   const Size: u8 = 2;

   pub fn new(king: Square, rook: Square) -> Self {
      debug_assert!(rook.rank() == king.rank());
      if rook < king { Self::A } else { Self::H }
   }

   const fn index(self) -> usize {
      (self as u8) as usize
   }

   pub fn iter() -> impl Iterator<Item = Self> {
      (0 .. Self::Size).map(|i| unsafe { mem::transmute(i) })
   }
}

impl Table_Rule {

   pub fn new(real: bool) -> Self {

      const Def: Castle<Square> = Castle { king_to: Square::None, rook_to: Square::None };

      let mut castle = [[Def; Wing::Size as usize]; Side::Size as usize];

      for sd in Side::iter() {

         let rk = square::rank_side(0, sd);

         for wing in Wing::iter() {

            let c = Castle[real as usize][wing.index()];

            castle[sd.index()][wing.index()] = Castle {
               king_to: Square::new(c.king_to, rk).unwrap(),
               rook_to: Square::new(c.rook_to, rk).unwrap(),
            };
         }
      }

      Self { castle }
   }

   pub fn king_rook_to(&self, wing: Wing, sd: Side) -> (Square, Square) {
      let c = self.castle[sd.index()][wing.index()];
      (c.king_to, c.rook_to)
   }
}

pub fn conv_in(mv: Move, bd: &Board) -> Move {

   let from = mv.from();
   let to   = mv.to();

   let sd = bd.turn();

   if mv.is_king(bd)
   && (to - from).abs() == 0o20 // 2 files apart
   && bd.square_is_empty(to)
   {
      let fl = match Wing::new(from, to) {
         Wing::A => { assert!(to.file() == 2); 0 }, // 'c' -> 'a' file
         Wing::H => { assert!(to.file() == 6); 7 }, // 'g' -> 'h' file
      };
      let rk = square::rank_side(0, sd);

      let to = Square::new(fl, rk).unwrap();
      assert!(bd.square(to) == Some((Piece::Rook, sd)));

      Move::new(from, to)

   } else {

      mv
   }
}

pub fn conv_out(mv: Move, bd: &Board) -> Move {

   let from = mv.from();
   let to   = mv.to();

   let sd = bd.turn();

   if mv.is_castle(bd)
   && from.file() == 4 // 'e' file
   {
      let fl = match Wing::new(from, to) {
         Wing::A => { assert!(to.file() == 0); 2 }, // 'a' -> 'c' file
         Wing::H => { assert!(to.file() == 7); 6 }, // 'h' -> 'g' file
      };
      let rk = square::rank_side(0, sd);

      let to = Square::new(fl, rk).unwrap();
      assert!(bd.square_is_empty(to));

      Move::new(from, to)

   } else {

      mv
   }
}

