
// modules

use std::ops;

// types

pub type Size = u16;

// traits

pub trait Seq<'a> {

   type Item: 'a;

   fn iter     (&'a     self) -> impl Iterator<Item = &'a     Self::Item>;

   fn size(&self) -> Size;

   fn range(&self) -> ops::Range<Size> {
      0 .. self.size()
   }

   fn has(&'a self, x: Self::Item) -> bool
   where Self::Item: Copy + Eq {

      self.iter().any(|&y| y == x)
   }
}

pub trait Array {

   type Item;

   fn as_slice    (&    self) -> &    [Self::Item];
   fn as_mut_slice(&mut self) -> &mut [Self::Item];

   fn move_to_front(&mut self, i: impl Into<u64>) {
      let slice = self.as_mut_slice();
      slice[0 ..= i.into() as usize].rotate_right(1);
   }
}

pub trait Grow {

   type Item;

   fn clear(&mut self);
   fn add  (&mut self, x: Self::Item);

   fn copy(&mut self, src: impl IntoIterator<Item = Self::Item>) {
      self.clear();
      self.append(src);
   }

   fn append(&mut self, src: impl IntoIterator<Item = Self::Item>) {

      for x in src {
         self.add(x);
      }
   }
}

// functions

impl<'a, C: Array> Seq<'a> for C
where C::Item: 'a {

   type Item = C::Item;

   fn iter(&'a self) -> impl Iterator<Item = &'a Self::Item> {
      self.as_slice().iter()
   }

   fn size(&self) -> Size {
      self.as_slice().len().try_into().unwrap()
   }
}

impl<T> Array for [T] {

   type Item = T;

   fn as_slice    (&    self) -> &    [T] { self }
   fn as_mut_slice(&mut self) -> &mut [T] { self }
}

impl<T, const N: usize> Array for [T; N] {

   type Item = T;

   fn as_slice    (&    self) -> &    [T] { self }
   fn as_mut_slice(&mut self) -> &mut [T] { self }
}

impl<T> Array for Vec<T> {

   type Item = T;

   fn as_slice    (&    self) -> &    [T] { self }
   fn as_mut_slice(&mut self) -> &mut [T] { self }
}

impl<T> Grow for Vec<T> {

   type Item = T;

   fn clear(&mut self)       { self.clear() }
   fn add  (&mut self, x: T) { self.push(x) }
}

