
// modules

use super::{Array, Buffer, Grow};

use std::ops::{self, Index, IndexMut};

// types

pub struct Array_Grow<T, const Max: usize> {
   size: Size,
   data: Buffer<T, Max>,
}

type Size = u16;

// functions

impl<T, const Max: usize> Array_Grow<T, Max> {

   const Max: Size = Max as Size;

   pub fn new() -> Self {

      assert!(Size::try_from(Max).is_ok());

      Self {
         size: 0,
         data: Default::default(),
      }
   }

   pub fn clear(&mut self) {
      self.truncate(0);
   }

   pub fn truncate(&mut self, size: Size) {

      assert!(size <= self.size);

      for i in size .. self.size {
         unsafe { self.data.drop(i) }
      }

      self.size = size;
   }

   pub fn push(&mut self, x: T) {

      assert!(!self.is_full());

      unsafe { self.data.write(self.size, x) }
      self.size += 1;
   }

   pub fn pop(&mut self) -> Option<T> {

      (self.size != 0).then(|| {
         self.size -= 1;
         unsafe { self.data.read(self.size) }
      })
   }

   pub fn is_full(&self) -> bool {
      self.size == Self::Max
   }

   pub fn size(&self) -> Size {
      self.size
   }

   pub fn slice(&self, start: Size, end: Size) -> &[T] {
      debug_assert!(start <= end && end <= self.size);
      unsafe { self.data.slice(start, end) }
   }

   pub fn slice_mut(&mut self, start: Size, end: Size) -> &mut [T] {
      debug_assert!(start <= end && end <= self.size);
      unsafe { self.data.slice_mut(start, end) }
   }
}

impl<T, const Max: usize> Default for Array_Grow<T, Max> {

   fn default() -> Self {
      Self::new()
   }
}

impl<T: Clone, const Max: usize> Clone for Array_Grow<T, Max> {

   fn clone(&self) -> Self {
      self.iter().cloned().collect()
   }
}

impl<T, const Max: usize> FromIterator<T> for Array_Grow<T, Max> {

   fn from_iter<I: IntoIterator<Item = T>>(iter: I) -> Self {

      let mut res = Self::new();
      res.extend(iter);
      res
   }
}

impl<T, const Max: usize> Extend<T> for Array_Grow<T, Max> {

   fn extend<I: IntoIterator<Item = T>>(&mut self, iter: I) {

      for x in iter {
         self.push(x);
      }
   }
}

impl<T, const Max: usize> Index<Size> for Array_Grow<T, Max> {

   type Output = T;

   fn index(&self, i: Size) -> &Self::Output {
      debug_assert!(i < self.size);
      unsafe { self.data.index(i) }
   }
}

impl<T, const Max: usize> IndexMut<Size> for Array_Grow<T, Max> {

   fn index_mut(&mut self, i: Size) -> &mut Self::Output {
      debug_assert!(i < self.size);
      unsafe { self.data.index_mut(i) }
   }
}

impl<T, const Max: usize> ops::Deref for Array_Grow<T, Max> {

   type Target = [T];

   fn deref(&self) -> &Self::Target {
      self.as_slice()
   }
}

impl<T, const Max: usize> ops::DerefMut for Array_Grow<T, Max> {

   fn deref_mut(&mut self) -> &mut Self::Target {
      self.as_mut_slice()
   }
}

impl<T, const Max: usize> Drop for Array_Grow<T, Max> {

   fn drop(&mut self) {
      self.clear();
   }
}

impl<T, const Max: usize> Array for Array_Grow<T, Max> {

   type Item = T;

   fn as_slice(&self) -> &[T] {
      unsafe { self.data.slice(0, self.size) }
   }

   fn as_mut_slice(&mut self) -> &mut [T] {
      unsafe { self.data.slice_mut(0, self.size) }
   }
}

impl<T, const Max: usize> Grow for Array_Grow<T, Max> {

   type Item = T;

   fn clear(&mut self) {
      self.clear();
   }

   fn add(&mut self, x: T) {
      self.push(x);
   }
}

