
// modules

use std::{mem, ops};

use mem::{MaybeUninit};

// types

pub struct Buffer<T, const Size: usize> {
   data: [MaybeUninit<T>; Size],
}

type Size = u16;

// functions

impl<T, const Size: usize> Buffer<T, { Size }> {

   pub const Size: Size = Size as Size;

   pub fn new() -> Self {

      assert!(Size::try_from(Size).is_ok());

      Self {
         data: [const { MaybeUninit::uninit() }; Size],
      }
   }

   pub unsafe fn read(&mut self, i: Size) -> T { // consume
      unsafe { self.ref_mut(i).assume_init_read() }
   }

   pub unsafe fn write(&mut self, i: Size, x: T) { // assume uninitialised
      unsafe { self.ref_mut(i).write(x); }
   }

   pub unsafe fn drop(&mut self, i: Size) {
      unsafe { self.ref_mut(i).assume_init_drop() }
   }

   pub unsafe fn index(&self, i: Size) -> &T {
      unsafe { self.ref_(i).assume_init_ref() }
   }

   pub unsafe fn index_mut(&mut self, i: Size) -> &mut T {
      unsafe { self.ref_mut(i).assume_init_mut() }
   }

   pub unsafe fn slice(&self, start: Size, end: Size) -> &[T] {

      let range = self.range(start, end);
      let slice = &self.data[range];
      unsafe { mem::transmute::<_, &[T]>(slice) }
   }

   pub unsafe fn slice_mut(&mut self, start: Size, end: Size) -> &mut [T] {

      let range = self.range(start, end);
      let slice = &mut self.data[range];
      unsafe { mem::transmute::<_, &mut [T]>(slice) }
   }

   fn range(&self, start: Size, end: Size) -> ops::Range<usize> {
      debug_assert!(start <= end && end <= Self::Size);
      start.into() .. end.into()
   }

   unsafe fn ref_(&self, i: Size) -> &MaybeUninit<T> {
      debug_assert!(i < Self::Size);
      unsafe { self.data.get_unchecked(i as usize) }
   }

   unsafe fn ref_mut(&mut self, i: Size) -> &mut MaybeUninit<T> {
      debug_assert!(i < Self::Size);
      unsafe { self.data.get_unchecked_mut(i as usize) }
   }
}

impl<T, const Size: usize> Default for Buffer<T, { Size }> {

   fn default() -> Self {
      Self::new()
   }
}

