
// modules

use super::{Array, Array_Grow, Grow};

use std::ops;

// types

#[derive(Clone)]
pub struct Stack_2<T, const Max: usize> {
   fp:   Size,
   data: Array_Grow<T, Max>,
}

type Size = u16;

// functions

impl<T, const Max: usize> Stack_2<T, Max> {

   pub fn new() -> Self {

      Self {
         fp:   0,
         data: Default::default(),
      }
   }

#[must_use]
   pub fn enter(&mut self) -> Size {

      let old_fp = self.fp;

      debug_assert!(self.sp() >= self.fp);
      self.fp = self.sp();

      old_fp
   }

   pub fn leave(&mut self, old_fp: Size) {

      debug_assert!(old_fp <= self.fp);

      self.clear();
      debug_assert!(self.fp == self.sp());

      self.fp = old_fp;
      debug_assert!(self.sp() >= self.fp);
   }

   pub fn clear(&mut self) {
      self.data.truncate(self.fp); // drops elements
   }

   pub fn push(&mut self, x: T) {
      self.data.push(x);
   }

   pub fn range(&self) -> ops::Range<Size> {
      0 .. self.size()
   }

   pub fn size(&self) -> Size {
      self.sp() - self.fp
   }

   pub fn sp(&self) -> Size {
      self.data.size()
   }
}

impl<T, const Max: usize> Default for Stack_2<T, Max> {

   fn default() -> Self {
      Self::new()
   }
}

impl<T, const Max: usize> ops::Index<Size> for Stack_2<T, Max> {

   type Output = T;

   fn index(&self, i: Size) -> &Self::Output {
      debug_assert!(i < self.size());
      &self.data[self.fp + i]
   }
}

impl<T, const Max: usize> ops::IndexMut<Size> for Stack_2<T, Max> {

   fn index_mut(&mut self, i: Size) -> &mut Self::Output {
      debug_assert!(i < self.size());
      &mut self.data[self.fp + i]
   }
}

impl<T, const Max: usize> ops::Deref for Stack_2<T, Max> {

   type Target = [T];

   fn deref(&self) -> &Self::Target {
      self.as_slice()
   }
}

impl<T, const Max: usize> ops::DerefMut for Stack_2<T, Max> {

   fn deref_mut(&mut self) -> &mut Self::Target {
      self.as_mut_slice()
   }
}

impl<T, const Max: usize> Array for Stack_2<T, Max> {

   type Item = T;

   fn as_slice(&self) -> &[T] {
      self.data.slice(self.fp, self.sp())
   }

   fn as_mut_slice(&mut self) -> &mut [T] {
      self.data.slice_mut(self.fp, self.sp())
   }
}

impl<T, const Max: usize> Grow for Stack_2<T, Max> {

   type Item = T;

   fn clear(&mut self) {
      self.clear();
   }

   fn add(&mut self, x: T) {
      self.push(x);
   }
}

