
// modules

use std::collections::{VecDeque};
use std::sync::{atomic, Condvar, Mutex};
use atomic::{AtomicU32, Ordering::*};

// types

pub struct Queue<T: Send> {
   data:  Mutex<Pair<T>>,
   empty: Condvar,
}

struct Pair<T> {
   queue:  VecDeque<T>,
   closed: bool,
}

pub struct AtomicF32(AtomicU32); // HACK

// functions

impl<T: Send> Queue<T> {

   pub fn new() -> Self {

      let data = Pair {
         queue:  Default::default(),
         closed: false,
      };

      Self {
         data:  data.into(),
         empty: Condvar::new(),
      }
   }

   pub fn put(&self, x: T) {

      let mut data = self.data.lock().unwrap();

      data.queue.push_back(x);
      self.empty.notify_one();
   }

   pub fn get(&self) -> Option<T> { // None => done

      let mut data = self.data.lock().unwrap();

      loop {

         match data.queue.pop_front() {

            Some(x) => break Some(x),

            None => { // empty
               if data.closed { break None }
               data = self.empty.wait(data).unwrap();
            }
         }
      }
   }

   pub fn peek(&self) -> Option<Option<T>> // None => empty; Some(None) => done
   where T: Clone
   {
      let data = self.data.lock().unwrap();

      match data.queue.front() {
         Some(x) => Some(Some(x.clone())),
         None    => if data.closed { Some(None) } else { None }
      }
   }

   pub fn close(&self) {

      let mut data = self.data.lock().unwrap();

      data.closed = true;
      self.empty.notify_all();
   }
}

impl AtomicF32 {

   pub fn new(x: f32) -> Self {
      Self(x.to_bits().into())
   }

   pub fn load(&self) -> f32 {
      f32::from_bits(self.0.load(Relaxed))
   }

   pub fn store(&self, x: f32) {
      self.0.store(x.to_bits(), Relaxed)
   }
}

impl Default for AtomicF32 {

   fn default() -> Self {
      Self::new(0.0)
   }
}

impl From<f32> for AtomicF32 {

   fn from(x: f32) -> Self {
      Self::new(x)
   }
}

